# Load necessary library
library(GA)
library(ggplot2)

# ==========================================
# 1. SETUP THE ARENA (Common Data)
# ==========================================
set.seed(24) # Fixed seed for fairness
num_cities <- 20
map_limit <- 100

# Generate Cities
cities <- data.frame(id = 1:num_cities, x = runif(num_cities, 0, map_limit), y = runif(num_cities, 0, map_limit))


##### empty space to load an extrnal file from me #####

# outfile <- "cities.csv"
# cities <- read.csv(outfile)

#######################################################


dist_matrix <- as.matrix(dist(cities[, c("x", "y")]))

# Helper: Calculate Total Distance of a Tour
calculate_tour_distance <- function(tour) {
  d <- 0
  for(i in 1:(length(tour)-1)) {
    d <- d + dist_matrix[tour[i], tour[i+1]]
  }
  d <- d + dist_matrix[tour[length(tour)], tour[1]] # Return to start
  return(d)
}

# ==========================================
# 2. CONTENDER 1: CUSTOM GA
# ==========================================
cat("Running Custom GA...\n")
start_time_custom <- Sys.time()

# Parameters
pop_size <- 80
generations <- 800
mutation_rate <- 0.1
elitism <- round(pop_size*0.1)

# A. Initialize
population <- t(replicate(pop_size, sample(1:num_cities)))

# B. Helper: Swap Mutation
mutate_swap <- function(tour) {
  if(runif(1) < mutation_rate) {
    idx <- sample(1:num_cities, 2)
    tour[idx] <- tour[rev(idx)]
  }
  return(tour)
}

# B. Helper: Inversion Mutation
mutate_inversion <- function(tour) {
  if(runif(1) < mutation_rate) {
    # 1. Select two random cut points
    idx <- sort(sample(1:num_cities, 2))
    
    # 2. Reverse the sequence of cities between the cut points
    tour[idx[1]:idx[2]] <- rev(tour[idx[1]:idx[2]])
  }
  return(tour)
}

# C. Helper: Order Crossover (OX1) - Essential for TSP!
crossover_ox1 <- function(p1, p2) {
  # 1. Select random slice
  cut_pts <- sort(sample(1:num_cities, 2))
  child <- rep(NA, num_cities)
  child[cut_pts[1]:cut_pts[2]] <- p1[cut_pts[1]:cut_pts[2]]
  
  # 2. Fill remaining from p2 (preserving order)
  p2_vals <- p2[!p2 %in% child] # Values in p2 that are NOT in the slice
  
  # Fill positions after the second cut, wrapping around
  fill_indices <- c((cut_pts[2]+1):num_cities, 1:(cut_pts[1]-1))
  # Handle edge cases where cut is at the end/start
  fill_indices <- fill_indices[fill_indices <= num_cities & fill_indices > 0]
  
  # If simple filling fails (edge cases), just fill NAs in order
  if(length(fill_indices) != length(p2_vals)) {
    child[is.na(child)] <- p2_vals
  } else {
    child[fill_indices] <- p2_vals
  }
  return(child)
}


# C. Helper: Partially Mapped Crossover (PMX)
crossover_pmx <- function(p1, p2) {
  # 1. Select two random cut points
  cut_pts <- sort(sample(1:num_cities, 2))
  child <- rep(NA, num_cities)
  
  # 2. Copy the swath (matching section) from p1 to the child
  child[cut_pts[1]:cut_pts[2]] <- p1[cut_pts[1]:cut_pts[2]]
  
  # 3. Create mapping segments
  swath_p1 <- p1[cut_pts[1]:cut_pts[2]]
  swath_p2 <- p2[cut_pts[1]:cut_pts[2]]
  
  # 4. Fill the rest of the child using p2 and the mapping
  for (i in 1:num_cities) {
    # Only fill empty spots (outside the crossover swath)
    if (is.na(child[i])) {
      candidate <- p2[i]
      
      # If the candidate from p2 is already in the child's swath, 
      # use the mapping to find a valid replacement
      while (candidate %in% swath_p1) {
        # Find where the candidate is in swath_p1
        idx <- which(swath_p1 == candidate)
        # Map it to the corresponding element in swath_p2
        candidate <- swath_p2[idx]
      }
      
      # Assign the valid candidate to the child
      child[i] <- candidate
    }
  }
  return(child)
}



# D. Main Loop
best_dist_history <- numeric(generations)

for(g in 1:generations) {
  # Fitness: Negative distance (because we want to maximize fitness)
  dists <- apply(population, 1, calculate_tour_distance)
  fitness <- -dists
  
  best_dist_history[g] <- min(dists)
  
  # Elitism: Keep best 2
  ranked_idx <- order(fitness, decreasing = TRUE)
  new_pop <- matrix(NA, nrow=pop_size, ncol=num_cities)
  new_pop[1:elitism, ] <- population[ranked_idx[1:elitism], ]
  
  # Breed rest
  for(i in (elitism+1):pop_size) {
    # Tournament Selection
    p1 <- population[sample(1:pop_size, 1), ]
    p2 <- population[sample(1:pop_size, 1), ]
    
    # Crossover & Mutation
    child <- crossover_ox1(p1, p2)
    # child <- crossover_pmx(p1, p2)
    # child <- crossover_erx(p1, p2)
    
    child <- mutate_swap(child)
    # child <- mutate_inversion(child)
    
    new_pop[i, ] <- child
  }
  population <- new_pop
}

custom_best_tour <- population[1, ] # Approximation of best
custom_min_dist <- min(dists)
custom_time <- Sys.time() - start_time_custom

# ==========================================
# 3. CONTENDER 2: GA LIBRARY
# ==========================================
cat("Running Library GA...\n")
start_time_lib <- Sys.time()

# Fitness function for library (Maximize negative distance)
lib_fitness <- function(tour) {
  return(-calculate_tour_distance(tour))
}

ga_result <- ga(
  type = "permutation",
  fitness = lib_fitness,
  lower = 1, upper = num_cities,
  popSize = pop_size, 
  maxiter = generations,
  pmutation = mutation_rate,
  elitism = elitism,
  monitor = FALSE
)

lib_best_tour <- as.numeric(ga_result@solution[1, ])
lib_min_dist <- -ga_result@fitnessValue
lib_time <- Sys.time() - start_time_lib

# ==========================================
# 4. THE VERDICT (Comparison)
# ==========================================

# Print Stats
cat("\n--- FINAL RESULTS ---\n")
cat(sprintf("Custom GA: Best Dist = %.2f | Time = %.2f sec\n", custom_min_dist, custom_time))
cat(sprintf("Library GA: Best Dist = %.2f | Time = %.2f sec\n", lib_min_dist, lib_time))

# Plotting Comparison
par(mfrow=c(2, 2)) # Side by side plots


# Plot Custom
plot(cities$x, cities$y, main=paste("Custom GA\nDist:", round(custom_min_dist, 1)), 
     pch=19, col="red", xlab="X", ylab="Y")
lines(cities$x[c(custom_best_tour, custom_best_tour[1])], 
      cities$y[c(custom_best_tour, custom_best_tour[1])], col="blue", lwd=2)

# Plot Library
plot(cities$x, cities$y, main=paste("Library GA\nDist:", round(lib_min_dist, 1)), 
     pch=19, col="darkgreen", xlab="X", ylab="Y")
lines(cities$x[c(lib_best_tour, lib_best_tour[1])], 
      cities$y[c(lib_best_tour, lib_best_tour[1])], col="purple", lwd=2)

# Plot Custom
plot(cities$x, cities$y, main=paste("The cities' locations", round(custom_min_dist, 1)), 
     pch=19, col="red", xlab="X", ylab="Y")

plot(ga_result,main="Library GA's performance\nper generation")